﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;

namespace YumeNikkiRandomizer
{
    class Map : RPGDataFile
    {
        string filepath = "";
        string filename = "";
        int id = 0;
        
        int tileset = 1; // 01
        int mapWidth = 20; // 02
        int mapHeight = 15; // 03
        int wrapType = 0; // 0b
        bool useParallax = false; // 1f
        string parallaxName = ""; // 20
        bool horzLoop = false; // 21
        bool vertLoop = false; // 22
        bool horzAuto = false; // 23
        int horzSpeed = 0; // 24
        bool vertAuto = false; // 25
        int vertSpeed = 0; // 26
        bool useRandomGenerator = false; // 28, verbose only (2003)
        int generatorMode = 0; // 29, verbose only (2003)
        bool topLevel = false; // 2a, verbose only (2003)
        int generatorGranularity = 0; // 30, verbose only (2003)
        int generatorRoomWidth = 4; // 31, verbose only (2003)
        int generatorRoomHeight = 1; // 32, verbose only (2003)
        bool generatorSurroundWithWalls = true; // 33, verbose only (2003)
        bool generatorUseUpperWall = true; // 34, verbose only (2003)
        bool generatorUseFloorB = true; // 35, verbose only (2003)
        bool generatorUseFloorC = true; // 36, verbose only (2003)
        bool generatorUseObstacleB = true; // 37, verbose only (2003)
        bool generatorUseObstacleC = true; // 38, verbose only (2003)
        long[] generatorTileX; // 3c, verbose only (2003)
        long[] generatorTileY; // 3d, verbose only (2003)
        long[] generatorTileID; // 3e, verbose only (2003)
        int[][] layer1Tiles; // 47, getTilesString or verbose only
        int[][] layer2Tiles; // 48, getTilesString or verbose only
        List<Event> events; // 51
        int saveCount = 0; // 5b, verbose only
        
        static string myClass = "Map";
        Chunks chunks;
        
        List<string> clearedPagesList;
        
        #region // Constants //
        
        // CharSet patterns.
        public const int P_LEFT = 0;
        public const int P_MIDDLE = 1;
        public const int P_RIGHT = 2;
        
        // Animation types.
        public const int A_NORMALNOSTEP = 0;
        public const int A_NORMALSTEP = 1;
        public const int A_FIXDIRNOSTEP = 2;
        public const int A_FIXDIRSTEP = 3;
        public const int A_FIXGRAPHIC = 4;
        public const int A_SPIN = 5;
        
        // Event layers.
        public const int L_BELOW = 0;
        public const int L_SAME = 1;
        public const int L_ABOVE = 2;
        
        // Move types.
        public const int M_STATIONARY = 0;
        public const int M_RANDOM = 1;
        public const int M_UPDOWN = 2;
        public const int M_LEFTRIGHT = 3;
        public const int M_TOWARD = 4;
        public const int M_AWAY = 5;
        public const int M_CUSTOM = 6;
        
        // Move event steps.
        public const int MS_UP = 0;
        public const int MS_RIGHT = 0;
        public const int MS_DOWN = 0;
        public const int MS_LEFT = 0;
        public const int MS_TOWARD = 9;
        public const int MS_AWAY = 10;
        public const int MS_WAIT = 23;
        public const int MS_FIXDIR = 26;
        public const int MS_UNFIXDIR = 27;
        public const int MS_SOUND = 35;
        public const int MS_STOPANIM = 38;
        public const int MS_STARTANIM = 39;
        public const int MS_TRANSUP = 40;
        public const int MS_TRANSDOWN = 41;
        #endregion
        
        public Map(string filepath)
        {
            loadFile(filepath);
        }
        public Map()
        {
        }
        
        // Loads a single .lmu map. Returns success.
        public bool loadFile(string filepath)
        {
            if (!File.Exists(filepath))
            {
                Console.WriteLine("Warning: Map file " + filepath + " not found! Base game may be missing something.");
                return false;
            }
            
            this.filepath = filepath;
            filename = Path.GetFileName(filepath);
            M.currentFile = filename;
            int.TryParse(filename.Replace("Map", "").Replace(".lmu", ""), out id);
            
            M.debugMessage("Loading " + filename + "...");
            
            FileStream f = File.OpenRead(filepath);
            
            try
            {
                chunks = new Chunks(f, myClass);
                
                M.stringCheck(f, "LcfMapUnit");
                
                if (chunks.next(0x01))
                    tileset = M.readLengthMultibyte(f);
                if (chunks.next(0x02))
                    mapWidth = M.readLengthMultibyte(f);
                if (chunks.next(0x03))
                    mapHeight = M.readLengthMultibyte(f);
                
                if (chunks.next(0x0b))
                    wrapType = M.readLengthMultibyte(f);
                
                if (chunks.next(0x1f))
                    useParallax = M.readLengthBool(f);
                if (chunks.next(0x20))
                    parallaxName = M.readString(f, M.S_FILENAME);
                if (chunks.next(0x21))
                    horzLoop = M.readLengthBool(f);
                if (chunks.next(0x22))
                    vertLoop = M.readLengthBool(f);
                if (chunks.next(0x23))
                    horzAuto = M.readLengthBool(f);
                if (chunks.next(0x24))
                    horzSpeed = M.readLengthMultibyte(f);
                if (chunks.next(0x25))
                    vertAuto = M.readLengthBool(f);
                if (chunks.next(0x26))
                    vertSpeed = M.readLengthMultibyte(f);
                
                if (chunks.next(0x28))
                    useRandomGenerator = M.readLengthBool(f);
                if (chunks.next(0x29))
                    generatorMode = M.readLengthMultibyte(f);
                if (chunks.next(0x2a))
                    topLevel = M.readLengthBool(f);
                if (chunks.next(0x30))
                    generatorGranularity = M.readLengthMultibyte(f);
                if (chunks.next(0x31))
                    generatorRoomWidth = M.readLengthMultibyte(f);
                if (chunks.next(0x32))
                    generatorRoomHeight = M.readLengthMultibyte(f);
                if (chunks.next(0x33))
                    generatorSurroundWithWalls = M.readLengthBool(f);
                if (chunks.next(0x34))
                    generatorUseUpperWall = M.readLengthBool(f);
                if (chunks.next(0x35))
                    generatorUseFloorB = M.readLengthBool(f);
                if (chunks.next(0x36))
                    generatorUseFloorC = M.readLengthBool(f);
                if (chunks.next(0x37))
                    generatorUseObstacleB = M.readLengthBool(f);
                if (chunks.next(0x38))
                    generatorUseObstacleC = M.readLengthBool(f);
                
                if (chunks.next(0x3c))
                    generatorTileX = M.readFourByteArray(f);
                if (chunks.next(0x3d))
                    generatorTileY = M.readFourByteArray(f);
                if (chunks.next(0x3e))
                    generatorTileID = M.readFourByteArray(f);
                
                if (chunks.next(0x47))
                    layer1Tiles = M.readTwoByteArray2D(f, mapHeight, mapWidth);
                if (chunks.next(0x48))
                    layer2Tiles = M.readTwoByteArray2D(f, mapHeight, mapWidth);
                
                if (chunks.next(0x51))
                    events = M.readList<Event>(f);
                
                if (chunks.next(0x5b))
                    saveCount = M.readLengthMultibyte(f);
                
                M.byteCheck(f, 0x00);
                
                f.Close();
            }
            catch (Exception e)
            {
                M.debugMessage(e.StackTrace);
                M.debugMessage(e.Message);
                Console.WriteLine("Aborting due to error.");
                
                f.Close();
                return false;
            }
            
            return true;
        }
        
        // Looks for unobtained items, and recursively searches exits.
        public void searchForProgress(string section = "", int depth = 0, string mapChain = "")
        {
            if (M.simulationMapsChecked.Contains(id + "|" + section))
                return;
            
            M.simulationMapsChecked.Add(id + "|" + section);
            
            // Pitch black maps without Lamp aren't required in logic.
            if (M.darknessLevel == 2 && M.mapsWithDarkness.Contains(id) && !M.simulationEffects.Contains(M.EF_LAMP))
                return;
            
            // Search for new Effects this map can give (when in the current section).
            if (M.effectKeysForMap.ContainsKey(id))
            {
                foreach (string key in M.effectKeysForMap[id])
                {
                    if (M.effectLocations[key] is MapLocation)
                    {
                        MapLocation location = M.effectLocations[key] as MapLocation;
                        if (location.section.Equals(section))
                            tryToGetEffect(key, section, depth, mapChain);
                    }
                    else if (M.effectLocations[key] is CombinedLocation)
                    {
                        CombinedLocation combo = M.effectLocations[key] as CombinedLocation;
                        foreach (MapLocation location in combo.locations)
                        {
                            if (location.section.Equals(section))
                                tryToGetEffect(key, section, depth, mapChain);
                        }
                    }
                }
            }
            
            // Search for new Essences this map can give (when in the current section).
            if (M.essenceKeysForMap.ContainsKey(id))
            {
                foreach (string key in M.essenceKeysForMap[id])
                {
                    if (M.essenceLocations[key] is MapLocation)
                    {
                        MapLocation location = M.essenceLocations[key] as MapLocation;
                        if (location.section.Equals(section) || location.mapID == M.MAP_MONO001) // Ignore Left/Right section of Monoko's tunnel
                            tryToGetEssence(key, depth, mapChain);
                    }
                    else if (M.essenceLocations[key] is CombinedLocation)
                    {
                        CombinedLocation combo = M.essenceLocations[key] as CombinedLocation;
                        foreach (MapLocation location in combo.locations)
                        {
                            if (location.section.Equals(section))
                                tryToGetEssence(key, depth, mapChain);
                        }
                    }
                }
            }
            
            // Search any maps accessible from this one, in this section.
            if (M.warpKeysForMap.ContainsKey(id))
            {
                foreach (string key in M.warpKeysForMap[id])
                {
                    if (M.warpLocations[key] is MapLocation)
                    {
                        MapLocation location = M.warpLocations[key] as MapLocation;
                        bool monokoException = false;
                        if (location.mapID == M.MAP_MONO001 && M.simulationEffects.Contains(M.EF_KNIFE)) // Knife allows you to get past Monoko to other side
                            monokoException = true;
                        
                        if (location.section.Equals(section) || monokoException)
                            tryToTakeExit(key, section, depth, mapChain);
                    }
                    else if (M.warpLocations[key] is CombinedLocation)
                    {
                        CombinedLocation combo = M.warpLocations[key] as CombinedLocation;
                        foreach (MapLocation location in combo.locations)
                        {
                            if (location.section.Equals(section))
                                tryToTakeExit(key, section, depth, mapChain);
                        }
                    }
                }
            }
            
            // Check dream bed warp if one exists in this map.
            if (M.mapsWithDreamBeds.Contains(id))
            {
                bool canBedWarp = false;
                if (M.derandomizeEvents) // Must be the predetermined bed ID
                {
                    int bedID = M.mapsWithDreamBeds.IndexOf(id) + 1;
                    if (M.fixedDreamBedID == bedID)
                        canBedWarp = true;
                }
                else // Possible for it to be this one
                    canBedWarp = true;
                
                if (canBedWarp)
                {
                    Warp bedWarp = M.exits["DreamBedCommon/StairsInBed"];
                    int destMap = bedWarp.mapID;
                    M.maps[destMap].searchForProgress(bedWarp.section, depth + 1, (mapChain != ""? mapChain + " -> " : "") + M.mapDisplayNames[id]);
                }
            }
        }
        
        // Adds any currently-accessible Effects on this map to simulation list.
        void tryToGetEffect(string locationKey, string section, int depth, string mapChain)
        {
            int effect = M.effects[locationKey];
            
            if (M.findingItemLocationsByDepth)
            {
                if (!M.effectLocationsAtDepth.ContainsKey(depth))
                {
                    M.effectLocationsAtDepth[depth] = new List<string>();
                    M.effectLocationMapChains[depth] = new List<string>();
                }
                if (!M.effectLocationsAtDepth[depth].Contains(locationKey))
                {
                    M.effectLocationsAtDepth[depth].Add(locationKey);
                    M.effectLocationMapChains[depth].Add(mapChain + " -> " + M.mapDisplayNames[id] + (section != ""? " (" + section + ")" : ""));
                }
                
                if (!M.simulationEffects.Contains(effect) && !M.pendingEffects.Contains(effect))
                {
                    M.pendingEffects.Add(effect);
                    M.simulationProgressMade = true;
                }
                return;
            }
            
            if (!M.simulationEffects.Contains(effect) && !M.pendingEffects.Contains(effect))
            {
                if (M.spoilerLogEnabled && !M.simulationProgressMade) // First item obtained in this loop, so add a 0 to indicate break
                    M.itemOrderForLog.Add(0);
                
                if (!M.spoilerLogEnabled) // Without need for spoiler log, can add items directly to make for fewer loops
                    M.simulationEffects.Add(effect);
                else // Otherwise, it's more readable to go one step at a time
                {
                    M.pendingEffects.Add(effect);
                    M.itemOrderForLog.Add(effect);
                }
                
                M.simulationProgressMade = true;
            }
        }
        
        // Adds any currently-accessible Essences on this map to simulation list.
        void tryToGetEssence(string locationKey, int depth, string mapChain)
        {
            int essence = M.essences[locationKey];
            
            // Only one Essence has a requirement that isn't rooted in "get to the map where it is," so it's just handled here.
            if (essence == M.ES_MONOKO && !M.simulationEffects.Contains(M.EF_STOPLIGHT))
                return;
            
            if (M.findingItemLocationsByDepth)
            {
                if (!M.essenceLocationsAtDepth.ContainsKey(depth))
                {
                    M.essenceLocationsAtDepth[depth] = new List<string>();
                    M.essenceLocationMapChains[depth] = new List<string>();
                }
                if (!M.essenceLocationsAtDepth[depth].Contains(locationKey))
                {
                    M.essenceLocationsAtDepth[depth].Add(locationKey);
                    M.essenceLocationMapChains[depth].Add(mapChain + " -> " + M.mapDisplayNames[id]);
                }
                
                if (!M.simulationEssences.Contains(essence) && !M.pendingEssences.Contains(essence))
                {
                    M.pendingEssences.Add(essence);
                    M.simulationProgressMade = true;
                }
                return;
            }
            
            if (!M.simulationEssences.Contains(essence) && !M.pendingEssences.Contains(essence))
            {
                if (M.spoilerLogEnabled && !M.simulationProgressMade) // First item obtained in this loop, so add a 0 to indicate break
                    M.itemOrderForLog.Add(0);
                
                if (!M.spoilerLogEnabled) // Without need for spoiler log, can add items directly to make for fewer loops
                    M.simulationEssences.Add(essence);
                else // Otherwise, it's more readable to go one step at a time
                {
                    M.pendingEssences.Add(essence);
                    M.itemOrderForLog.Add(essence);
                }
                
                M.simulationProgressMade = true;
            }
        }
        
        // Explores any currently-accessible exits on this map, checking any requirements to take that exit.
        void tryToTakeExit(string exitName, string section, int depth, string mapChain)
        {
            switch (id)
            {
                case M.MAP_NEXUS: // Need specified effect for Nexus locks
                    int lockIndex = -1;
                    switch (exitName)
                    {
                        case "DoorRoom/01Frog": lockIndex = 0; break;
                        case "DoorRoom/02HatAndScarf": lockIndex = 1; break;
                        case "DoorRoom/03Umbrella": lockIndex = 2; break;
                        case "DoorRoom/04Knife": lockIndex = 3; break;
                        case "DoorRoom/05YukiOnna": lockIndex = 4; break;
                        case "DoorRoom/06Hairstyle": lockIndex = 5; break;
                        case "DoorRoom/07Bicycle": lockIndex = 6; break;
                        case "DoorRoom/08EyeballHand": lockIndex = 7; break;
                        case "DoorRoom/09Small": lockIndex = 8; break;
                        case "DoorRoom/10CatEars": lockIndex = 9; break;
                        case "DoorRoom/11Neon": lockIndex = 10; break;
                        case "DoorRoom/12Lamp": lockIndex = 11; break;
                    }
                    if (lockIndex != -1)
                    {
                        int requiredEffect = M.nexusLocks[lockIndex];
                        if (requiredEffect != 0 && !M.simulationEffects.Contains(requiredEffect))
                            return;
                    }
                    break;
                
                case M.MAP_SNOWWORLD: // Need Knife to stab Toriningen before it'll warp you
                    if (exitName.Equals("05YukiOnna-Toriningen/RedA-Stuck") && !M.simulationEffects.Contains(M.EF_KNIFE))
                        return;
                    break;
                
                case M.MAP_NUMBERWORLD:
                    switch (exitName)
                    {
                        case "12Lamp-GuardedDoor/LampSmallRoom1": // Need Knife or Cat to get past NPC in front of door
                            if (!M.simulationEffects.Contains(M.EF_KNIFE) && !M.simulationEffects.Contains(M.EF_CAT))
                                return;
                            break;
                        case "12Lamp-CanalDoor/LampSmallRoom3-WallDoor": // Isolated by canal, only accessible if you came out of it
                            if (!section.Equals("Canal Door"))
                                return;
                            break;
                        case "12Lamp-Zipper/LampSmallRoom4": // Knife needed to open zipper
                            if (!M.simulationEffects.Contains(M.EF_KNIFE))
                                return;
                            break;
                    }
                    break;
                
                case M.MAP_ROOMOFBEDS: // Need Knife to stab Toriningen and activate closet warp
                    if (exitName.StartsWith("LampSmallRoom2-ThirdCloset/RedB") && !M.simulationEffects.Contains(M.EF_KNIFE))
                        return;
                    break;
                
                case M.MAP_HELL: // Need Knife for Henkei Shita random warp
                    if (exitName.StartsWith("RedA-Henkei/FootprintsA-Henkei") && !M.simulationEffects.Contains(M.EF_KNIFE))
                        return;
                    break;
                
                case M.MAP_GUTTER004: // Need Small to get through pipes (though not to come back out)
                    if (exitName.Equals("Gutter_004-Small/Gutter_005-Left") && !M.simulationEffects.Contains(M.EF_SMALL))
                        return;
                    break;
                
                case M.MAP_MONO:
                    switch (exitName)
                    {
                        case "Mono-LoopTunnelRight/Mono002-Right":
                        case "Mono-LoopTunnelLeft/Mono003-TunnelRight":
                            if (!M.simulationEffects.Contains(M.EF_KNIFE)) // Need Knife to stab Dave Spector and activate loop tunnel
                                return;
                            break;
                        case "Mono-EyeBox/Mono005":
                            if (!M.simulationEffects.Contains(M.EF_KNIFE)) // Need Knife for eyeball box random warp
                                return;
                            break;
                    }
                    break;
                
                case M.MAP_BLAZINGCORRIDOR: // Need Umbrella or Yuki-Onna to put out flames
                    if (exitName.Equals("OutsideStorage(Flames)/Storeroom")
                     && !M.simulationEffects.Contains(M.EF_UMBRELLA) && !M.simulationEffects.Contains(M.EF_YUKIONNA))
                        return;
                    break;
                
                case M.MAP_MARS05: // Need Small to get into basement
                    if (exitName.Equals("Mars_05-Small/MarsBasement_01") && !M.simulationEffects.Contains(M.EF_SMALL))
                        return;
                    break;
                
                case M.MAP_MARSBASEMENTSTAIRS: // Need Small to get out of basement
                    if (exitName.Equals("MarsBasement_01/Mars_05-Small") && !M.simulationEffects.Contains(M.EF_SMALL))
                        return;
                    break;
                
                case M.MAP_MALLROOF: // Need Witch to access flight scene
                    if (exitName.Equals("DepartmentRoof/FlyWithBroom") && !M.simulationEffects.Contains(M.EF_WITCH))
                        return;
                    break;
            }
            
            Warp exit = M.exits[exitName];
            int destMap = exit.mapID;
            string exitDesc = "";
            if (id == M.MAP_NEXUS && M.nexusExitNames.ContainsKey(exitName))
                exitDesc = " (" + M.nexusExitNames[exitName] + ")";
            M.maps[destMap].searchForProgress(exit.section, depth + 1, (mapChain != ""? mapChain + " -> " : "") + M.mapDisplayNames[id] + exitDesc);
        }
        
        // Makes changes to data using the validated random layout.
        public void applyLayout()
        {
            // Apply starting settings.
            if (id == M.MAP_GAMESTART)
                applyGameStartSettings();
            
            // Edit Effects.
            foreach (string key in M.effectLocations.Keys)
            {
                if (M.effectLocations[key] is MapLocation)
                {
                    MapLocation location = M.effectLocations[key] as MapLocation;
                    if (location.mapID == id)
                    {
                        editEffect(location, M.effects[key]);
                        reflectEffect(location, key);
                    }
                }
                else if (M.effectLocations[key] is CombinedLocation)
                {
                    CombinedLocation combo = M.effectLocations[key] as CombinedLocation;
                    foreach (MapLocation location in combo.locations)
                    {
                        if (location.mapID == id)
                        {
                            editEffect(location, M.effects[key]);
                            reflectEffect(location, key);
                        }
                    }
                }
            }
            
            // Edit warps.
            clearedPagesList = new List<string>();
            
            foreach (string key in M.warpLocations.Keys)
            {
                if (M.warpLocations[key] is MapLocation)
                {
                    MapLocation location = M.warpLocations[key] as MapLocation;
                    if (location.mapID == id)
                        editWarp(location, M.exits[key]);
                }
                else if (M.warpLocations[key] is CombinedLocation)
                {
                    CombinedLocation combo = M.warpLocations[key] as CombinedLocation;
                    foreach (MapLocation location in combo.locations)
                    {
                        if (location.mapID == id)
                            editWarp(location, M.exits[key]);
                    }
                }
            }
            
            // Edit Nexus locks.
            if (id == M.MAP_NEXUS)
            {
                for (int i = 0; i < 12; i++)
                {
                    int lockingEffect = M.nexusLocks[i];
                    if (lockingEffect != 0)
                    {
                        string charSet = "";
                        int charIndex = 1, dir = M.D_DOWN, pattern = P_MIDDLE, animationType = A_FIXGRAPHIC, frequency = 3;
                        switch (lockingEffect)
                        {
                            case M.EF_FROG:
                                charSet = "Event_002"; charIndex = 5; animationType = A_NORMALNOSTEP; frequency = 1;
                                break;
                            case M.EF_HATANDSCARF:
                                charSet = "Event_002"; charIndex = 1; dir = M.D_RIGHT; animationType = A_FIXGRAPHIC;
                                break;
                            case M.EF_UMBRELLA:
                                charSet = "Event_002"; charIndex = 2; dir = M.D_RIGHT; pattern = P_RIGHT; animationType = A_FIXGRAPHIC; // Standing up
                                break;
                            case M.EF_YUKIONNA:
                                charSet = "Event_004"; charIndex = 3; animationType = A_NORMALSTEP;
                                break;
                            case M.EF_KNIFE:
                                charSet = "Event_004"; charIndex = 1; dir = M.D_LEFT; animationType = A_FIXGRAPHIC;
                                break;
                            case M.EF_EYEBALLHAND:
                                charSet = "Event_003"; charIndex = 1; animationType = A_FIXDIRSTEP;
                                break;
                            case M.EF_FAT:
                                charSet = "Event_032"; charIndex = 3; dir = M.D_RIGHT; animationType = A_FIXDIRSTEP; // Bottom half
                                break;
                            case M.EF_SMALL:
                                charSet = "Event_004"; charIndex = 7; animationType = A_NORMALNOSTEP; frequency = 4;
                                break;
                            case M.EF_FLUTE:
                                charSet = "Event_004"; charIndex = 2; dir = M.D_LEFT; animationType = A_FIXGRAPHIC;
                                break;
                            case M.EF_NEON:
                                charSet = "Event_002"; charIndex = 7; animationType = A_NORMALSTEP; frequency = 2;
                                break;
                            case M.EF_NOPPERABOU:
                                charSet = "Event_001"; charIndex = 6; animationType = A_NORMALSTEP; frequency = 2;
                                break;
                            case M.EF_SEVEREDHEAD:
                                charSet = "Event_002"; charIndex = 2; pattern = P_MIDDLE; animationType = A_SPIN;
                                break;
                            case M.EF_TOWEL:
                                charSet = "Event_004"; charIndex = 6; animationType = A_NORMALSTEP;
                                break;
                            case M.EF_CAT:
                                charSet = "Event_003"; charIndex = 3; animationType = A_SPIN;
                                break;
                            case M.EF_LAMP:
                                charSet = "Event_004"; charIndex = 4; animationType = A_NORMALNOSTEP; frequency = 2;
                                break;
                            case M.EF_BICYCLE:
                                charSet = "Event_002"; charIndex = 1; dir = M.D_DOWN; pattern = P_RIGHT; animationType = A_FIXGRAPHIC;
                                break;
                            case M.EF_LONGHAIR:
                                charSet = "Event_003"; charIndex = 4; animationType = A_NORMALSTEP;
                                break;
                            case M.EF_POOPHAIR:
                                charSet = "Event_003"; charIndex = 5; animationType = A_NORMALSTEP;
                                break;
                            case M.EF_BLONDEHAIR:
                                charSet = "Event_003"; charIndex = 6; animationType = A_NORMALSTEP;
                                break;
                            case M.EF_TRIANGLEKERCHIEF:
                                charSet = "Event_003"; charIndex = 7; animationType = A_NORMALSTEP;
                                break;
                            case M.EF_WITCH:
                                charSet = "Event_020"; charIndex = 4; dir = M.D_RIGHT; animationType = A_FIXDIRSTEP; frequency = 1;
                                break;
                            case M.EF_ONI:
                                charSet = "[FC]Event_001"; charIndex = 1; animationType = A_NORMALNOSTEP; frequency = 2;
                                break;
                            case M.EF_SQUISHY:
                                charSet = "[FC]Event_002"; charIndex = 3; animationType = A_FIXDIRSTEP; frequency = 1;
                                break;
                            case M.EF_STOPLIGHT:
                                charSet = "Event_020"; charIndex = 5; animationType = A_FIXGRAPHIC;
                                break;
                        }
                        
                        // Edit the corresponding "obtained, so don't try to push Madotsuki out of this lock anymore" switch in event 38.
                        // First fork is EventID == 42, second fork is Switch [Obtained] == ON, repeating that way for each lock.
                        editForkSwitch(38, 1, M.effectObtainedSwitch[lockingEffect], (i + 1) * 2);
                        
                        // Edit the graphic on first page (present) and condition for second page (gone).
                        editPageAnimation(42 + i, 1, charSet, charIndex - 1, dir, pattern, animationType, frequency);
                        editPageSwitchCondition(42 + i, 2, M.effectObtainedSwitch[lockingEffect], 2);
                        
                        // Add location hint if enabled.
                        if (M.nexusLockHints)
                        {
                            foreach (string key in M.effectLocations.Keys)
                            {
                                if (M.effects[key] == lockingEffect)
                                {
                                    insertWindowMessage(42 + i, 1, 0, M.centeredText(M.effectLocationText[key]));
                                    break;
                                }
                            }
                        }
                        
                        // For Strober, create a new event two tiles up for top half with same conditions.
                        if (lockingEffect == M.EF_FAT)
                        {
                            Event bottom = getEventWithID(42 + i);
                            int newEvent = addEvent(bottom.eventX, bottom.eventY - 2, 2);
                            editPageAnimation(newEvent, 1, charSet, charIndex - 1, M.D_UP, pattern, animationType, frequency); // Facing up = top half
                            editPageSwitchCondition(newEvent, 2, M.effectObtainedSwitch[lockingEffect]);
                            editPageLayer(newEvent, 1, L_ABOVE);
                        }
                    }
                    else // Set condition to Switch 262 (NexusLocks) so that it either won't appear at all (if Nexus locks are off), or will prioritize Page 2
                        editPageSwitchCondition(42 + i, 2, 262, 2);
                }
            }
            
            // Miscellaneous changes.
            if (id == M.MAP_HELL && M.heckificationType == 1) // Heck mode: Replace Hell maze tiles with tiles from Map0047b
                setTileLayer1(M.heckTiles);
            
            if (id == M.MAP_WARPMAZE && M.easyTeleporterMaze)
            {
                editTeleport(3, 1, new Warp(M.MAP_WARPMAZE, 178, 170, M.D_SAME)); // Make left warp from entry point go to end warp
                editTeleport(4, 1, new Warp(M.MAP_WARPMAZE, 178, 170, M.D_SAME)); // Make right warp from entry point go to end warp
            }
            
            if (M.darknessLevel == 2 && M.mapsWithDarkness.Contains(id)) // Pitch Black: Create events to block all entrances in dark maps without Lamp
            {
                int blockCount = id == M.MAP_DARKWORLD? 6
                               : id == M.MAP_INSIDESHACKA? 1
                               : id == M.MAP_JUKAI3? 4
                               : id == M.MAP_FCDUNGEON001? 14
                               : 0;
                
                // Create events with a solid page when lights are off, and an empty page when using Lamp.
                int[] blockingEvents = new int[blockCount];
                for (int i = 0; i < blockingEvents.Length; i++)
                {
                    blockingEvents[i] = addEvent(0, 0, 2);
                    editPageLayer(blockingEvents[i], 2, L_SAME);
                    editPageSwitchCondition(blockingEvents[i], 2, 150); // Lights Out switch
                }
                
                // Place events according to map.
                switch (id)
                {
                    case M.MAP_DARKWORLD:
                        editEventPosition(blockingEvents[0], 11, 11); // Left of Nexus door
                        editEventPosition(blockingEvents[1], 12, 12); // Below Nexus door
                        editEventPosition(blockingEvents[2], 13, 11); // Right of Nexus door
                        
                        editEventPosition(blockingEvents[3], 48, 72); // Left of sand gate
                        editEventPosition(blockingEvents[4], 49, 73); // Below sand gate
                        editEventPosition(blockingEvents[5], 50, 72); // Right of sand gate
                        break;
                    
                    case M.MAP_INSIDESHACKA:
                        editEventPosition(blockingEvents[0], 27, 17); // Left of narrow entrance
                        break;
                    
                    case M.MAP_JUKAI3:
                        editEventPosition(blockingEvents[0], 2, 9); // Right of top exit
                        editEventPosition(blockingEvents[1], 1, 10); // Right-lower of top exit
                        
                        editEventPosition(blockingEvents[2], 2, 44); // Right of bottom exit
                        editEventPosition(blockingEvents[3], 1, 45); // Right-lower of bottom exit
                        break;
                    
                    case M.MAP_FCDUNGEON001:
                        editEventPosition(blockingEvents[0], 7, 4); // Top-right of stairs to village
                        editEventPosition(blockingEvents[1], 5, 4); // Top-left of top-left stairs to village
                        editEventPosition(blockingEvents[2], 4, 5); // Left of top-left stairs to village
                        editEventPosition(blockingEvents[3], 5, 6); // Bottom-left of top-left stairs to village
                        editEventPosition(blockingEvents[4], 6, 7); // Below top-left stairs to village
                        editEventPosition(blockingEvents[5], 7, 6); // Bottom-right of top-left stairs to village
                        
                        editEventPosition(blockingEvents[6], 121, 64); // Up-right of stairs to dungeon 2/glitch event dungeon 2
                        editEventPosition(blockingEvents[7], 122, 65); // Right of stairs to dungeon 2/glitch event dungeon 2
                        
                        editEventPosition(blockingEvents[8], 32, 76); // Top-left of bottom stairs to dungeon 3
                        editEventPosition(blockingEvents[9], 34, 76); // Top-right of bottom stairs to dungeon 3
                        editEventPosition(blockingEvents[10], 35, 77); // Right of bottom stairs to dungeon 3
                        editEventPosition(blockingEvents[11], 34, 78); // Bottom-right of bottom stairs to dungeon 3
                        editEventPosition(blockingEvents[12], 33, 79); // Below bottom stairs to dungeon 3
                        editEventPosition(blockingEvents[13], 32, 78); // Bottom-left of bottom stairs to dungeon 3
                        break;
                }
            }
            
            if (id == M.MAP_FCBASEMENT016 && M.derandomizeEvents) // Move top mushroom out of the way just so their random movement isn't an issue
                editEventPosition(3, 5, 10);
            if (id == M.MAP_FCLIZARDVILLAGE && M.derandomizeEvents) // Move villager out of the way and make them stationary for similar reasons
            {
                editEventPosition(1, 25, 3);
                editPageMovement(1, 1, M_STATIONARY);
                editPageMovement(1, 2, M_STATIONARY);
            }
        }
        
        // Edits settings in game start event.
        void applyGameStartSettings()
        {
            // Variable #1: REM9Version
            // Variable #2: Nasu high score (to beat for essence)
            // --------------------
            editSwitchValue(1, 5, M.nexusLockCount > 0, 1); // NexusLocks
            editSwitchValue(1, 5, M.bicyclePlacement == 2, 2); // StartWithBicycle
            editSwitchValue(1, 5, M.eyeballPlacement == 2, 3); // StartWithEyeball
            // Switch #4: Obtained Bicycle (set to true if StartWithBicycle is true)
            // Switch #5: Possess Bicycle (set to true if StartWithBicycle is true)
            // Switch #6: Obtained Eyeball-hand (set to true if StartWithEyeball is true)
            // Switch #7: Possess Eyeball-hand (set to true if StartWithEyeball is true)
            // --------------------
            editVariableValue(1, 5, M.effectsGoal, 3); // EffectGoal
            editVariableValue(1, 5, M.essencesGoal, 4); // EssenceGoal
            editStatUpValue(1, 5, M.essencesGoal, 1); // Increase max MP to show Essence goal
            // --------------------
            editSwitchValue(1, 5, M.fixChairGlitch, 8); // FixChairGlitch
            editVariableValue(1, 5, M.walkSpeed, 5); // WalkSpeed
            editVariableValue(1, 5, M.bikeSpeed, 6); // BikeSpeed
            editSwitchValue(1, 5, M.derandomizeEvents, 9); // DerandomizeEvents
            editVariableValue(1, 5, 80 + M.random.Next(155), 7); // Random starting X for falling Nasu eggplants (80 to 234)
            editVariableValue(1, 5, 5 + M.random.Next(11), 8); // RedNasuFixedCount (originally 1/50, now 5~15)
            // Variable #9: RedNasuCounter
            editVariableValue(1, 5, 1 + M.random.Next(15), 10); // UboaFixedCount (originally 1/64, now 1~15)
            // Variable #11: UboaCounter
            editVariableValue(1, 5, 1 + M.random.Next(5), 12); // ShipCrashFixedCount (originally 1/6, now 1~5)
            // Variable #13: ShipCrashCounter
            editVariableValue(1, 5, M.fixedDreamBedID, 14); // DreamBedFixedID
            editVariableValue(1, 5, 1 + M.random.Next(30), 15); // HellStabFixedCount (originally 1/128, now 1~30)
            // Variable #16: HellStabCounter
            editVariableValue(1, 5, M.fixedClosetWarpID, 17); // ClosetWarpID
            editVariableValue(1, 5, M.fixedMiniHellClosetID, 18); // MiniHellClosetID
            editVariableValue(1, 5, M.fixedPurpleVersion, 19); // PurpleVersion
            editVariableValue(1, 5, M.fixedPurpleClosetSwap, 20); // PurpleClosetSwap
            editVariableValue(1, 5, M.fixedVillageWarper, 21); // EarthVillageWarper
            editVariableValue(1, 5, 1 + M.random.Next(10), 22); // EyeBoxFixedCount (originally 1/30, now 1~10)
            // Variable #23: EyeBoxCounter
            editSwitchValue(1, 5, M.fixedGlitchEvent, 10); // GlitchEventAlways
        }
        
        // Edits Effect-get commands to give a different one.
        void editEffect(MapLocation location, int effect)
        {
            int newObtainedSwitch = M.effectObtainedSwitch[effect];
            int newPossessedSwitch = M.effectPossessedSwitch[effect];
            bool fc = M.mapsInFCWorld.Contains(id);
            
            if (!fc)
            {
                string newMessage = M.centeredText(M.effectProperName[effect]);
                editForkSwitch(location.eventID, location.page, newObtainedSwitch);
                editMessage(location.eventID, location.page, newMessage);
                editSwitch(location.eventID, location.page, newObtainedSwitch, true, 1);
                editSwitch(location.eventID, location.page, newPossessedSwitch, true, 2);
                editItem(location.eventID, location.page, effect);
            }
            else
            {
                string newPicture = "FCMessageWindow(" + M.effectFilename[effect] + ")";
                editForkSwitch(location.eventID, location.page, newObtainedSwitch, 2);
                editPicture(location.eventID, location.page, newPicture);
                editSwitch(location.eventID, location.page, newObtainedSwitch, true, 3);
                editSwitch(location.eventID, location.page, newPossessedSwitch, true, 4);
                editItem(location.eventID, location.page, effect);
            }
        }
        
        // Edits graphics and pages to make Effect-givers reflect the new Effect.
        void reflectEffect(MapLocation location, string locationKey)
        {
            if (!M.giversReflectEffect)
                return;
            
            int effect = M.effects[locationKey];
            
            // If this is the regular location for the Effect, do nothing.
            if (locationKey.Equals(M.effectProperName[effect]))
                return;
            
            bool fc = M.mapsInFCWorld.Contains(id);
            
            // Prepare standard settings and pages before changing what's unique for the specific Effect.
            string charSet = "";
            int charIndex = 1, dir = M.D_DOWN, pattern = P_MIDDLE, animationType = A_FIXGRAPHIC, frequency = 3;
            
            PageCustomRoute beckonRoute = new PageCustomRoute(new MoveStep(MS_FIXDIR), new MoveStep(MS_STOPANIM),
                new MoveStep(MS_TOWARD), new MoveStep(MS_TOWARD), new MoveStep(MS_STARTANIM));
            
            PageCustomRoute smallKnifeRoute = null;
            if (effect == M.EF_KNIFE)
            {
                smallKnifeRoute = new PageCustomRoute();
                for (int i = 1; i <= 12; i++)
                {
                    smallKnifeRoute.addStep(new MoveStep(MS_SOUND, 10, i < 7? 100 : 120, 50, "Footsteps_007"));
                    smallKnifeRoute.addStep(new MoveStep(MS_RIGHT));
                    if (i == 6)
                        smallKnifeRoute.addStep(new MoveStep(MS_UP));
                    else if (i == 12)
                        smallKnifeRoute.addStep(new MoveStep(MS_DOWN));
                    else
                        smallKnifeRoute.addStep(new MoveStep(MS_RIGHT));
                }
            }
            
            MoveRoute stabbedRoute = new MoveRoute();
            for (int i = 1; i <= 4; i++)
            {
                if (i == 1)
                    stabbedRoute.addStep(new MoveStep(MS_STOPANIM));
                else
                    stabbedRoute.addStep(new MoveStep(MS_WAIT));
                stabbedRoute.addStep(new MoveStep(MS_TRANSUP));
                stabbedRoute.addStep(new MoveStep(MS_TRANSUP));
            }
            
            Page beckonPage = new Page();
            beckonPage.conditions.setSwitch1(152);
            beckonPage.setMoveType(M_CUSTOM, 8, beckonRoute);
            beckonPage.setEventLayer(L_SAME, true);
            
            Page stopPage = new Page();
            stopPage.conditions.setSwitch1(153);
            stopPage.setMoveType(M_STATIONARY);
            stopPage.setEventLayer(L_SAME);
            stopPage.commands.Clear();
            stopPage.commands.Add(new Command(Command.C_CALLEVENT, 0, new int[] { 1, 10005, 1 })); // Fixed event, this event, page 1
            stopPage.commands.Add(new Command(Command.C_TERMINATOR, 0, new int[] { }));
            
            Page knifePage = new Page();
            knifePage.conditions.setSwitch1(89);
            knifePage.setEventLayer(L_SAME);
            knifePage.commands.Clear();
            knifePage.commands.Add(new Command(Command.C_CALLEVENT, 0, new int[] { 0, 67, 1 })); // Common event, 67
            if (!fc)
                knifePage.commands.Add(new Command(Command.C_SHOWANIMATION, 0, new int[] { 29, 10005, 0, 0 })); // Animation 29 (BloodSpray), this event
            else
                knifePage.commands.Add(new Command(Command.C_FLASHEVENT, 0, new int[] { 10005, 31, 0, 0, 31, 10, 0 })); // This event, 31 red, 31 power, 1 sec
            knifePage.commands.Add(new Command(Command.C_CALLEVENT, 0, new int[] { 0, 68, 1 })); // Common event, 68
            knifePage.commands.Add(new Command(Command.C_MOVEEVENT, 0, 10005, 7, false, false, stabbedRoute)); // Fade away this event
            knifePage.commands.Add(new Command(Command.C_MOVEALL, 0, new int[] { }));
            knifePage.commands.Add(new Command(Command.C_VARIABLE, 0, new int[] { 0, 20, 0, 0, 3, 1, 8 })); // Variable 20 = random 1~8
            knifePage.commands.Add(new Command(Command.C_FORK, 0, new int[] { 1, 20, 0, 1, 0 })); // If Variable 20 == 1
            knifePage.commands.Add(new Command(Command.C_SHOWANIMATION, 1, new int[] { 34, 10005, 1, 0 })); // Animation 34 (100Yen), this event, wait
            knifePage.commands.Add(new Command(Command.C_CHANGEMONEY, 1, new int[] { 0, 0, 100 })); // Add 100 money
            knifePage.commands.Add(new Command(Command.C_BLANK, 1, new int[] { }));
            knifePage.commands.Add(new Command(Command.C_FORKEND, 0, new int[] { }));
            knifePage.commands.Add(new Command(Command.C_CALLEVENT, 0, new int[] { 0, 112, 1 })); // Common event, 112
            knifePage.commands.Add(new Command(Command.C_ERASEEVENT, 0, new int[] { }));
            knifePage.commands.Add(new Command(Command.C_TERMINATOR, 0, new int[] { }));
           
            Page menuPage = null, menuStopPage = null;
            
            switch (effect)
            {
                case M.EF_FROG:
                    charSet = "Event_002"; charIndex = 5; animationType = A_NORMALNOSTEP; frequency = 1;
                    beckonPage.setAnimation(charSet, charIndex - 1, dir, pattern, animationType, 2);
                    knifePage.setAnimation(charSet, charIndex - 1, dir, pattern, animationType, frequency);
                    knifePage.setMoveType(M_RANDOM, 3);
                    stopPage.setAnimation(charSet, charIndex - 1, dir, pattern, animationType, frequency);
                    break;
                case M.EF_HATANDSCARF:
                    charSet = "Event_002"; charIndex = 1; dir = M.D_RIGHT; animationType = A_FIXGRAPHIC;
                    beckonPage = null;
                    knifePage = null;
                    stopPage = null;
                    break;
                case M.EF_UMBRELLA:
                    charSet = "Event_002"; charIndex = 2; dir = M.D_RIGHT; pattern = P_RIGHT; animationType = A_FIXGRAPHIC; // Standing up
                    beckonPage = null;
                    knifePage = null;
                    stopPage = null;
                    break;
                case M.EF_YUKIONNA:
                    charSet = "Event_004"; charIndex = 3; animationType = A_NORMALSTEP;
                    beckonPage.setAnimation(charSet, charIndex - 1, dir, pattern, animationType, 2);
                    knifePage.setAnimation(charSet, charIndex - 1, dir, pattern, animationType, frequency);
                    knifePage.setMoveType(M_AWAY, 3);
                    stopPage.setAnimation(charSet, charIndex - 1, dir, pattern, A_NORMALNOSTEP, frequency);
                    break;
                case M.EF_KNIFE:
                    charSet = "Event_004"; charIndex = 1; dir = M.D_LEFT; animationType = A_FIXGRAPHIC;
                    beckonPage = null;
                    knifePage = null;
                    stopPage = null;
                    break;
                case M.EF_EYEBALLHAND:
                    charSet = "Event_003"; charIndex = 1; animationType = A_FIXDIRSTEP;
                    beckonPage = null;
                    knifePage.setAnimation(charSet, charIndex - 1, dir, pattern, animationType, frequency);
                    knifePage.setMoveType(M_STATIONARY);
                    stopPage.setAnimation(charSet, charIndex - 1, dir, pattern, A_NORMALNOSTEP, frequency);
                    break;
                case M.EF_FAT:
                    charSet = "Event_032"; charIndex = 3; dir = M.D_UP; animationType = A_FIXDIRSTEP; // Top half
                    beckonPage.setAnimation(charSet, charIndex - 1, dir, pattern, animationType, 6);
                    beckonPage.setMoveType(M_STATIONARY);
                    knifePage.setAnimation(charSet, charIndex - 1, dir, pattern, animationType, 1);
                    knifePage.setMoveType(M_STATIONARY);
                    stopPage.setAnimation(charSet, charIndex - 1, dir, pattern, A_FIXGRAPHIC, frequency);
                    break;
                case M.EF_SMALL:
                    charSet = "Event_004"; charIndex = 7; animationType = A_NORMALNOSTEP; frequency = 4;
                    beckonPage.setAnimation(charSet, charIndex - 1, dir, pattern, A_NORMALSTEP, 2);
                    knifePage.setAnimation(charSet, charIndex - 1, dir, pattern, animationType, frequency);
                    knifePage.setMoveType(M_CUSTOM, 8, smallKnifeRoute);
                    stopPage.setAnimation(charSet, charIndex - 1, dir, pattern, animationType, frequency);
                    break;
                case M.EF_FLUTE:
                    charSet = "Event_004"; charIndex = 2; dir = M.D_LEFT; animationType = A_FIXGRAPHIC;
                    beckonPage = null;
                    knifePage = null;
                    stopPage = null;
                    break;
                case M.EF_NEON:
                    charSet = "Event_002"; charIndex = 7; animationType = A_NORMALSTEP; frequency = 2;
                    beckonPage.setAnimation(charSet, charIndex - 1, dir, pattern, animationType, frequency);
                    knifePage.setAnimation(charSet, charIndex - 1, dir, pattern, animationType, frequency);
                    knifePage.setMoveType(M_AWAY, 6);
                    stopPage.setAnimation(charSet, charIndex - 1, dir, pattern, A_NORMALNOSTEP, 3);
                    break;
                case M.EF_NOPPERABOU:
                    charSet = "Event_001"; charIndex = 6; animationType = A_NORMALSTEP; frequency = 2;
                    beckonPage.setAnimation(charSet, charIndex - 1, dir, pattern, animationType, frequency);
                    knifePage.setAnimation(charSet, charIndex - 1, dir, pattern, animationType, frequency);
                    knifePage.setMoveType(M_AWAY, 7);
                    stopPage.setAnimation(charSet, charIndex - 1, dir, pattern, A_NORMALNOSTEP, frequency);
                    break;
                case M.EF_SEVEREDHEAD:
                    charSet = "Event_002"; charIndex = 2; pattern = P_MIDDLE; animationType = A_SPIN;
                    beckonPage = null;
                    knifePage = null;
                    stopPage.setAnimation(charSet, charIndex - 1, dir, pattern, A_NORMALNOSTEP, frequency);
                    break;
                case M.EF_TOWEL:
                    charSet = "Event_004"; charIndex = 6; animationType = A_NORMALSTEP;
                    beckonPage.setAnimation(charSet, charIndex - 1, dir, pattern, A_SPIN, 2);
                    knifePage.setAnimation(charSet, charIndex - 1, dir, pattern, animationType, frequency);
                    knifePage.setMoveType(M_RANDOM, 3);
                    stopPage.setAnimation(charSet, charIndex - 1, dir, pattern, A_FIXDIRNOSTEP, frequency);
                    break;
                case M.EF_CAT:
                    charSet = "Event_003"; charIndex = 3; animationType = A_SPIN;
                    beckonPage.setAnimation(charSet, charIndex - 1, dir, pattern, animationType, 2);
                    knifePage.setAnimation(charSet, charIndex - 1, dir, pattern, animationType, frequency);
                    knifePage.setMoveType(M_RANDOM, 8);
                    stopPage.setAnimation(charSet, charIndex - 1, dir, pattern, A_NORMALNOSTEP, frequency);
                    break;
                case M.EF_LAMP:
                    charSet = "Event_004"; charIndex = 4; animationType = A_NORMALNOSTEP; frequency = 2;
                    beckonPage.setAnimation(charSet, charIndex - 1, dir, pattern, A_FIXDIRNOSTEP, frequency);
                    knifePage.setAnimation(charSet, charIndex - 1, dir, pattern, animationType, frequency);
                    knifePage.setMoveType(M_AWAY, 8);
                    stopPage.setAnimation(charSet, charIndex - 1, dir, pattern, animationType, frequency);
                    break;
                case M.EF_BICYCLE:
                    charSet = "Event_002"; charIndex = 1; dir = M.D_DOWN; pattern = P_RIGHT; animationType = A_FIXGRAPHIC;
                    beckonPage = null;
                    knifePage = null;
                    stopPage = null;
                    break;
                case M.EF_LONGHAIR:
                    charSet = "Event_003"; charIndex = 4; animationType = A_NORMALSTEP;
                    beckonPage.setAnimation(charSet, charIndex - 1, dir, pattern, animationType, 2);
                    knifePage.setAnimation(charSet, charIndex - 1, dir, pattern, animationType, frequency);
                    knifePage.setMoveType(M_AWAY, 6);
                    stopPage.setAnimation(charSet, charIndex - 1, dir, pattern, A_NORMALNOSTEP, frequency);
                    break;
                case M.EF_POOPHAIR:
                    charSet = "Event_003"; charIndex = 5; animationType = A_NORMALSTEP;
                    beckonPage.setAnimation(charSet, charIndex - 1, dir, pattern, animationType, 2);
                    knifePage.setAnimation(charSet, charIndex - 1, dir, pattern, animationType, frequency);
                    knifePage.setMoveType(M_AWAY, 6);
                    stopPage.setAnimation(charSet, charIndex - 1, dir, pattern, A_NORMALNOSTEP, frequency);
                    break;
                case M.EF_BLONDEHAIR:
                    charSet = "Event_003"; charIndex = 6; animationType = A_NORMALSTEP;
                    beckonPage.setAnimation(charSet, charIndex - 1, dir, pattern, animationType, 2);
                    knifePage.setAnimation(charSet, charIndex - 1, dir, pattern, animationType, frequency);
                    knifePage.setMoveType(M_AWAY, 6);
                    stopPage.setAnimation(charSet, charIndex - 1, dir, pattern, A_NORMALNOSTEP, frequency);
                    break;
                case M.EF_TRIANGLEKERCHIEF:
                    charSet = "Event_003"; charIndex = 7; animationType = A_NORMALSTEP;
                    beckonPage.setAnimation(charSet, charIndex - 1, dir, pattern, animationType, 2);
                    knifePage.setAnimation(charSet, charIndex - 1, dir, pattern, animationType, frequency);
                    knifePage.setMoveType(M_AWAY, 6);
                    stopPage.setAnimation(charSet, charIndex - 1, dir, pattern, A_NORMALNOSTEP, frequency);
                    break;
                case M.EF_WITCH:
                    charSet = "Event_020"; charIndex = 4; dir = M.D_RIGHT; animationType = A_FIXDIRSTEP; frequency = 1;
                    beckonPage = null;
                    knifePage.setAnimation(charSet, charIndex - 1, dir, pattern, animationType, frequency);
                    knifePage.setMoveType(M_STATIONARY);
                    stopPage.setAnimation(charSet, charIndex - 1, dir, pattern, A_FIXGRAPHIC, frequency);
                    break;
                case M.EF_ONI:
                    charSet = "[FC]Event_001"; charIndex = 1; animationType = A_NORMALNOSTEP; frequency = 2;
                    beckonPage.setAnimation(charSet, charIndex - 1, dir, pattern, animationType, frequency);
                    knifePage.setAnimation(charSet, charIndex - 1, dir, pattern, animationType, frequency);
                    knifePage.setMoveType(M_RANDOM, 1);
                    stopPage.setAnimation(charSet, charIndex - 1, dir, pattern, animationType, frequency);
                    
                    menuPage = new Page();
                    menuPage.conditions.setSwitch1(182);
                    menuPage.setAnimation(charSet, charIndex - 1, dir, pattern, animationType, 2);
                    menuPage.setEventLayer(L_SAME);
                    menuPage.setMoveType(M_STATIONARY);
                    break;
                case M.EF_SQUISHY:
                    charSet = "[FC]Event_002"; charIndex = 3; animationType = A_FIXDIRSTEP; frequency = 1;
                    beckonPage.setAnimation(charSet, charIndex - 1, dir, pattern, animationType, 2);
                    knifePage.setAnimation(charSet, charIndex - 1, dir, pattern, animationType, frequency);
                    knifePage.setMoveType(M_STATIONARY);
                    stopPage.setAnimation(charSet, charIndex - 1, M.D_RIGHT, pattern, A_FIXGRAPHIC, 3);
                    
                    menuPage = new Page();
                    menuPage.conditions.setSwitch1(182);
                    menuPage.setAnimation(charSet, charIndex - 1, dir, pattern, A_FIXDIRNOSTEP, 2);
                    menuPage.setEventLayer(L_SAME);
                    menuPage.setMoveType(M_STATIONARY);
                    
                    menuStopPage = new Page();
                    menuStopPage.conditions.setSwitch1(182);
                    menuStopPage.conditions.setSwitch2(153);
                    menuStopPage.setAnimation(charSet, charIndex - 1, M.D_RIGHT, pattern, A_FIXDIRNOSTEP, 2);
                    menuStopPage.setEventLayer(L_SAME);
                    menuStopPage.setMoveType(M_STATIONARY);
                    break;
                case M.EF_STOPLIGHT:
                    charSet = "Event_020"; charIndex = 5; animationType = A_FIXDIRNOSTEP; // Changed from Fixed Graphic so they animate while moving
                    beckonPage.setAnimation(charSet, charIndex - 1, dir, pattern, A_FIXDIRSTEP, 6);
                    knifePage = null;
                    stopPage = null;
                    break;
            }
            
            if (fc && menuPage == null) // Need to create a menuPage if in an FC map, so base it on Stoplight page
            {
                int menuAnimationType = A_FIXGRAPHIC, menuFrequency = 3;
                if (stopPage != null)
                {
                    menuAnimationType = stopPage.animationType;
                    menuFrequency = stopPage.animationFrequency;
                }
                
                menuPage = new Page();
                menuPage.conditions.setSwitch1(182);
                menuPage.setAnimation(charSet, charIndex - 1, dir, pattern, menuAnimationType, menuFrequency);
                menuPage.setEventLayer(L_SAME);
                menuPage.setMoveType(M_STATIONARY);
            }
            else if (!fc) // No need for these, even if it's Squishy-giver
            {
                menuPage = null;
                menuStopPage = null;
            }
            
            Event ev = getEventWithID(location.eventID);
            Page normalPage = ev.getPage(1);
            normalPage.setAnimation(charSet, charIndex - 1, dir, pattern, animationType, frequency);
            
            // Special case for Oni location: it explicitly faces hero, but this breaks fixed-direction graphics, so remove it in that case.
            if (location.mapID == M.MAP_FCBASEMENT019)
            {
                if (animationType == A_FIXDIRNOSTEP || animationType == A_FIXDIRSTEP || animationType == A_FIXGRAPHIC)
                    removeMoveCommand(location.eventID, location.page);
            }
            
            // Remove existing special pages.
            while (ev.getPageCount() > 1)
                ev.removePage(2);
           
            // Add non-null pages (with proper IDs) in order.
            int pageNum = 2;
            if (beckonPage != null)
            {
                beckonPage.pageNum = pageNum;
                ev.addPage(beckonPage);
                pageNum++;
            }
            if (knifePage != null)
            {
                knifePage.pageNum = pageNum;
                ev.addPage(knifePage);
                pageNum++;
            }
            if (stopPage != null)
            {
                stopPage.pageNum = pageNum;
                ev.addPage(stopPage);
                pageNum++;
            }
            if (menuPage != null)
            {
                menuPage.pageNum = pageNum;
                ev.addPage(menuPage);
                pageNum++;
            }
            if (menuStopPage != null)
            {
                menuStopPage.pageNum = pageNum;
                ev.addPage(menuStopPage);
                pageNum++;
            }
            
            // Finally, special cases for Strober. If Strober is being replaced, remove top half event.
            if (locationKey.Equals("Fat"))
                removeEvent(122);
            
            // If adding Strober... it'd be nice to add a top-half event that follows the bottom, but for now I'm just using the top-half graphic only.
        }
        
        // Edits warp commands to go somewhere different.
        void editWarp(MapLocation location, Warp warp)
        {
            // Only clear warp manager commons once. Otherwise, if there's two warps in one page, it'll remove the first warp's manager calls.
            if (!clearedPagesList.Contains(location.eventID + "|" + location.page))
            {
                removeWarpManagers(location.eventID, location.page);
                clearedPagesList.Add(location.eventID + "|" + location.page);
            }
            
            // Edit the teleport command.
            editTeleport(location.eventID, location.page, warp, location.commandNum);
            
            // Determine what warp manager common events are necessary for this warp, and add them before and after the teleport commnad.
            List<int> commonsBeforeTeleport = new List<int>();
            List<int> commonsAfterTeleport = new List<int>();
            determineWarpManagers(ref commonsBeforeTeleport, ref commonsAfterTeleport, location, warp);
            
            foreach (int commonID in commonsBeforeTeleport)
                addCommonNearTeleport(location, commonID, 0);
            foreach (int commonID in commonsAfterTeleport)
                addCommonNearTeleport(location, commonID, 1);
            
            // If entering/leaving FC World, change certain special source map transitions so that they properly hide the graphics change.
            int sourceMap = location.mapID;
            int destMap = warp.mapID;
            
            if ((M.mapsInFCWorld.Contains(sourceMap) && !M.mapsInFCWorld.Contains(destMap))
             || (!M.mapsInFCWorld.Contains(sourceMap) && M.mapsInFCWorld.Contains(destMap)))
            {
                if ((sourceMap == M.MAP_FORESTWORLD && location.eventID == 3) // Gate
                 || (sourceMap == M.MAP_FACECARPETPLAZA && (location.eventID == 1 || location.eventID == 2)) // Top/bottom gates
                 || (sourceMap == M.MAP_IGLOO7 && location.eventID == 2) // Pool
                 || (sourceMap == M.MAP_PINKSEA && location.eventID == 7)) // Return exit
                {
                    // Change first (pre-move) and second (post-move) Variable Set commands to 0.
                    editVariableValue(location.eventID, location.page, 0);
                    editVariableValue(location.eventID, location.page, 0, 2);
                }
                if ((sourceMap == M.MAP_MONO && location.eventID == 8) // Eye box
                 || (sourceMap == M.MAP_MONO005 && location.eventID == 27)) // Eye box
                {
                    // Change first erase transition to Fade Out. Change sixth Variable Set command (first before teleport) to 0.
                    editScreenTransitionEffect(location.eventID, location.page, 0); // 0 = Fade Out
                    editVariableValue(location.eventID, location.page, 0, 6);
                }
                if (sourceMap == M.MAP_FCPYRAMIDS && location.eventID == 1) // Warping oni
                {
                    // Change first Variable Set command to 0.
                    editVariableValue(location.eventID, location.page, 0);
                }
            }
        }
        
        // Determines what commons to call before and after a teleport command, given the location and destination.
        public static void determineWarpManagers(ref List<int> beforeTeleport, ref List<int> afterTeleport, CommandLocation location, Warp warp)
        {
            int sourceMap = 0;
            if (location is MapLocation)
                sourceMap = (location as MapLocation).mapID;
            int destMap = warp.mapID;
            
            // Exiting Special Area
            if (M.mapsInFCWorld.Contains(sourceMap) && !M.mapsInFCWorld.Contains(destMap))
                afterTeleport.Add(139); // FC World Exit
            if (M.mapsWithSnow.Contains(sourceMap) && !M.mapsWithSnow.Contains(destMap))
                beforeTeleport.Add(239); // [Pre]Exiting Snow
            if (M.mapsWithRain.Contains(sourceMap) && !M.mapsWithRain.Contains(destMap))
                beforeTeleport.Add(240); // [Pre]Exiting Rain
            if (M.mapsWithDarkness.Contains(sourceMap) && !M.mapsWithDarkness.Contains(destMap))
                afterTeleport.Add(241); // [Post]Exiting Dark
            
            // Entering Special Area
            if (!M.mapsInFCWorld.Contains(sourceMap) && M.mapsInFCWorld.Contains(destMap))
                afterTeleport.Add(138); // FC World Entrance
            if (sourceMap == M.MAP_NEXUS && destMap == M.MAP_NEXUS)
                afterTeleport.Add(242); // [Post]Nexus to Nexus
            if (sourceMap != M.MAP_NEXUS && destMap == M.MAP_NEXUS)
                afterTeleport.Add(235); // [Post]Entering Nexus
            if (M.mapsWithSnow.Contains(destMap))
                beforeTeleport.Add(236); // [Pre]Entering Snow
            if (M.mapsWithRain.Contains(destMap))
                beforeTeleport.Add(237); // [Pre]Entering Rain
            if (M.mapsWithDarkness.Contains(destMap))
                beforeTeleport.Add(238); // [Pre]Entering Dark
            
            // Miscellaneous
            if (warp.phaseForward)
                afterTeleport.Add(284); // [Post]Phase Forward
            if ((warp.mapID == M.MAP_DREAMROOM && warp.x == 13 && warp.y == 7)
             || (warp.mapID == M.MAP_BLOCKWORLD && warp.x == 42 && warp.y == 58)
             || (warp.mapID == M.MAP_SNOWWORLD && warp.x == 91 && warp.y == 49)
             || (warp.mapID == M.MAP_CANDLEWORLD && warp.x == 68 && warp.y == 87)
             || (warp.mapID == M.MAP_ROOMOFBEDS && warp.x == 31 && warp.y == 40))
            {
                beforeTeleport.Add(285); // [Pre]Exit Dream Bed
                afterTeleport.Add(286); // [Post]Exit Dream Bed
            }
            
            switch (destMap)
            {
                // Random Map Events
                case M.MAP_MALLA:
                    if (warp.extraInfo.Equals("One-Way Elevator")) // Special case for the Passage Outside Storeroom elevator warp
                        afterTeleport.Add(246); // [Post]Enter Mall A-1 (guaranteed for cone to be gone)
                    else
                        afterTeleport.Add(245); // [Post]Enter Mall A (random unless Derandomized Events is on)
                    break;
                case M.MAP_FCPYRAMIDS:
                    afterTeleport.Add(247); break; // [Post]Enter FC Small Map 002
                case M.MAP_BARRACKSSETTLEMENT:
                    beforeTeleport.Add(248); break; // [Pre]Enter Earth
                case M.MAP_GUTTER006:
                    beforeTeleport.Add(249); break; // [Pre]Enter Gutter_006
                case M.MAP_WITCHISLAND:
                    beforeTeleport.Add(250); break; // [Pre]Enter Shore
                case M.MAP_MALLROOF:
                    afterTeleport.Add(251); break; // [Post]Enter Mall Roof
                
                // Show Picture/BG
                case M.MAP_SPACESHIPENTRANCE:
                    beforeTeleport.Add(290); // [Pre]Unlanded Ship
                    afterTeleport.Add(255); // [Post]Enter Spaceship Entrance
                    break;
                case M.MAP_SPACESHIP01:
                    if (warp.x < 16)
                        afterTeleport.Add(256); // [Post]Enter Spaceship_01 (Left Side)
                    else
                        afterTeleport.Add(257); // [Post]Enter Spaceship_01 (Right Side)
                    break;
                case M.MAP_SPACESHIP02:
                    afterTeleport.Add(258); break; // [Post]Enter Spaceship_02/Exit
                case M.MAP_SPACESHIPEXIT:
                    beforeTeleport.Add(291); // [Pre]Landed Ship
                    afterTeleport.Add(258); // [Post]Enter Spaceship_02/Exit
                    break;
                case M.MAP_SPACESHIP03:
                    afterTeleport.Add(259); break; // [Post]Enter Spaceship_03
                case M.MAP_MONO006:
                    afterTeleport.Add(260); break; // [Post]Enter Mono006
                case M.MAP_STAIRWAYTOTHESKY:
                    afterTeleport.Add(261); break; // [Post]Enter Sand Stairs
                case M.MAP_PARKCLIFF:
                    afterTeleport.Add(262); break; // [Post]Enter Park Cliff
                case M.MAP_VERANDA:
                    afterTeleport.Add(263); break; // [Post]Enter Veranda
                case M.MAP_MONO010:
                    afterTeleport.Add(264); break; // [Post]Enter Mono010
                
                // Other Event Variables
                case M.MAP_HIGHWAYC:
                    beforeTeleport.Add(270); break; // [Pre]Enter Highway C
                case M.MAP_BLAZINGCORRIDOR:
                    if (warp.x > 14) // From right side, so flames depend on effect being used
                    {
                        beforeTeleport.Add(271); // [Pre]Enter Blazing Corridor
                        afterTeleport.Add(272); // [Post]Enter Blazing Corridor
                    }
                    else // From storeroom side, so flames are always gone
                    {
                        beforeTeleport.Add(275); // [Pre]Fire-Side Blazing Corridor
                        afterTeleport.Add(276); // [Post]Fire-Side Blazing Corridor
                    }
                    break;
                case M.MAP_INSIDEBRICKSA:
                    afterTeleport.Add(273); break; // [Post]Enter Inside Bricks A
                case M.MAP_INSIDEBRICKSC:
                    afterTeleport.Add(274); break; // [Post]Enter Inside Bricks C
                case M.MAP_MONO011:
                    if (warp.x == 17 && warp.y == 69) // On stairs, so allow walking on them
                        beforeTeleport.Add(277); // [Pre]Mono011 Stairs
                    break;
                
                // Miscellaneous
                case M.MAP_FACECARPETPLAZA:
                    if (warp.y > 25) // Coming in bottom entrance
                        afterTeleport.Add(280); // [Post]Phase Up
                    else // Coming in top entrance
                        afterTeleport.Add(281); // [Post]Phase Down
                    break;
                case M.MAP_FORESTWORLD:
                    if (warp.x == 93 && warp.y == 22) // "Inside" gate to Face Carpet Plaza
                    {
                        if (warp.dir == M.D_UP) // Facing up
                            afterTeleport.Add(280); // [Post]Phase Up
                        else // Facing down
                            afterTeleport.Add(281); // [Post]Phase Down
                    }
                    break;
                case M.MAP_BLOCKWORLD:
                    if (warp.x == 89 && warp.y == 71) // Coming out of White Desert gate, so swap Mafurako up and use upper tileset
                        afterTeleport.Add(282); // [Post]Upper Hat and Scarf
                    else // On ground, so use ground tileset
                        afterTeleport.Add(287); // [Post]Lower Hat and Scarf
                    break;
                case M.MAP_MARS05:
                    if (warp.x == 16 && warp.y == 13) // Coming up hill, so call UFO event
                        afterTeleport.Add(283); // [Post]Mars_05
                    break;
                case M.MAP_TRAININTERIOR:
                    if (warp.extraInfo.Equals("Pre-Trip")) // Coming in from "destination A" sets "current train stop" accordingly
                        beforeTeleport.Add(288); // [Pre]Train Pre-Trip
                    else // Coming in from "destination B" sets "current train stop" accordingly
                        beforeTeleport.Add(289); // [Pre]Train Post-Trip
                    break;
                case M.MAP_GUILLOTINEWORLDSMALL:
                case M.MAP_GUILLOTINEWORLDBIG:
                    afterTeleport.Add(292); // [Post]Enter Purple
                    break;
            }
        }
        
        // Edits switch ID in Xth conditional branch on specified page.
        void editForkSwitch(int eventID, int page, int newSwitch, int num = 1)
        {
            Command command = getCommand(eventID, page, num, Command.C_FORK);
            command.args[1] = newSwitch;
        }
        
        // Edits item ID in Xth item command on specified page.
        void editItem(int eventID, int page, int newItem, int num = 1)
        {
            Command command = getCommand(eventID, page, num, Command.C_CHANGEITEMS);
            command.args[2] = newItem;
        }
        
        // Edits message in Xth message command on specified page.
        void editMessage(int eventID, int page, string newMessage, int num = 1)
        {
            Command command = getCommand(eventID, page, num, Command.C_MESSAGE);
            command.stringArg = newMessage;
        }
        
        // Edits picture file in Xth picture command on specified page.
        void editPicture(int eventID, int page, string newPicture, int num = 1)
        {
            Command command = getCommand(eventID, page, num, Command.C_PICTURE);
            command.stringArg = newPicture;
        }
        
        // Edits warp in Xth teleport command on specified page.
        void editTeleport(int eventID, int page, Warp newWarp, int num = 1)
        {
            Command command = getCommand(eventID, page, num, Command.C_TELEPORT);
            command.args[0] = newWarp.mapID;
            command.args[1] = newWarp.x;
            command.args[2] = newWarp.y;
            command.args[3] = newWarp.dir + 1;
        }
        
        // Edits switch ID and value in Xth switch command on specified page.
        void editSwitch(int eventID, int page, int newSwitch, bool newValue = true, int num = 1)
        {
            Command command = getCommand(eventID, page, num, Command.C_SWITCH);
            command.args[1] = newSwitch;
            command.args[3] = newValue == true? 0 : 1;
        }
        
        // Edits just switch value in Xth switch command on specified page.
        void editSwitchValue(int eventID, int page, bool newValue, int num = 1)
        {
            Command command = getCommand(eventID, page, num, Command.C_SWITCH);
            command.args[3] = newValue == true? 0 : 1;
        }
        
        // Edits variable ID and value in Xth variable command on specified page.
        void editVariable(int eventID, int page, int newVariable, int newValue, int num = 1)
        {
            Command command = getCommand(eventID, page, num, Command.C_VARIABLE);
            command.args[1] = newVariable;
            command.args[5] = newValue;
        }
        
        // Edits just variable value in Xth variable command on specified page.
        void editVariableValue(int eventID, int page, int newValue, int num = 1)
        {
            Command command = getCommand(eventID, page, num, Command.C_VARIABLE);
            command.args[5] = newValue;
        }
        
        // Edits just value in Xth stat change command on specified page.
        void editStatUpValue(int eventID, int page, int newValue, int num = 1)
        {
            Command command = getCommand(eventID, page, num, Command.C_CHANGESTATS);
            command.args[5] = newValue;
        }
        
        // Edits just effect in Xth screen transition command on specified page.
        void editScreenTransitionEffect(int eventID, int page, int newValue, int num = 1)
        {
            Command command = getCommand(eventID, page, num, Command.C_TRANSITIONCHANGE);
            command.args[1] = newValue;
        }
        
        // Edits switch ID in specified page's trigger conditions.
        void editPageSwitchCondition(int eventID, int page, int newSwitch, int num = 1)
        {
            Page editPage = getEventWithID(eventID).getPage(page);
            if (num == 1)
                editPage.conditions.setSwitch1(newSwitch);
            else if (num == 2)
                editPage.conditions.setSwitch2(newSwitch);
            else
                Console.Write("Error: Attempting to edit invalid page condition!");
        }
        
        // Edits sprite and animation settings in specified page.
        void editPageAnimation(int eventID, int page, string charSet, int charIndex, int dir, int pattern, int animationType, int frequency)
        {
            Page editPage = getEventWithID(eventID).getPage(page);
            editPage.setAnimation(charSet, charIndex, dir, pattern, animationType, frequency);
        }
        
        // Edits movement settings in specified page.
        void editPageMovement(int eventID, int page, int movementType, int moveFrequency = 3)
        {
            Page editPage = getEventWithID(eventID).getPage(page);
            editPage.setMoveType(movementType, moveFrequency);
        }
        
        // Edits event layer in specified page.
        void editPageLayer(int eventID, int page, int layer, bool forbidOverlap = false)
        {
            Page editPage = getEventWithID(eventID).getPage(page);
            editPage.setEventLayer(layer, forbidOverlap);
        }
        
        // Edits coordinates of an event.
        void editEventPosition(int eventID, int x, int y)
        {
            Event ev = getEventWithID(eventID);
            ev.setPosition(x, y);
        }
        
        // Removes common calls related to warps, to be replaced by the appropriate ones for the new warp.
        void removeWarpManagers(int eventID, int page)
        {
            Page editPage = getEventWithID(eventID).getPage(page);
            
            List<Command> toRemove = new List<Command>();
            for (int i = 0; i < editPage.commands.Count; i++)
            {
                Command command = editPage.commands[i];
                if (command.opcode == Command.C_CALLEVENT && command.args[0] == 0)
                {
                    int commonID = command.args[1];
                    if ((commonID >= 234 && commonID <= 299) // New management commons
                      || commonID == 138 || commonID == 139) // Original FC World entrance/exit commons
                        toRemove.Add(command);
                }
            }
            
            foreach (Command command in toRemove)
                editPage.commands.Remove(command);
        }
        
        // Removes Xth move command on specified page.
        void removeMoveCommand(int eventID, int page, int num = 1)
        {
            Page editPage = getEventWithID(eventID).getPage(page);
            Command command = getCommand(eventID, page, num, Command.C_MOVEEVENT);
            editPage.commands.Remove(command);
        }
        
        // Generic function for adding a common call either before or after a teleport command.
        void addCommonNearTeleport(MapLocation location, int commonID, int offset)
        {
            Page editPage = getEventWithID(location.eventID).getPage(location.page);
            int commandNum = 0;
            
            for (int i = 0; i < editPage.commands.Count; i++)
            {
                Command command = editPage.commands[i];
                if (command.opcode == Command.C_TELEPORT)
                {
                    commandNum++;
                    if (commandNum == location.commandNum)
                    {
                        editPage.commands.Insert(i + offset, new Command(Command.C_CALLEVENT, command.indent, new int[] { 0, commonID, 0 }));
                        break;
                    }
                }
            }
        }
        
        // Adds a one-line message with window frame at given position.
        void insertWindowMessage(int eventID, int page, int commandIndex, string message)
        {
            Page editPage = getEventWithID(eventID).getPage(page);
            int indent = editPage.commands[commandIndex].indent;
            
            // Inserting commands in reverse order
            editPage.commands.Insert(commandIndex, new Command(Command.C_CALLEVENT, indent, new int[] { 0, 103, 0 })); // Window Picture Clear
            editPage.commands.Insert(commandIndex, new Command(Command.C_MESSAGE, indent, new int[] {}, message));
            editPage.commands.Insert(commandIndex, new Command(Command.C_CALLEVENT, indent, new int[] { 0, 102, 0 })); // One-Line Window Display
        }
        
        // Adds an event to the map and returns the event ID.
        int addEvent(int x, int y, int pageCount = 1)
        {
            int nextID = 1;
            bool verified = true;
            do
            {
                verified = true;
                for (int i = 0; i < events.Count; i++)
                {
                    if (events[i].eventNum == nextID)
                    {
                        nextID++;
                        verified = false;
                    }
                }
            } while (!verified);
            
            events.Add(new Event(nextID, "EV" + nextID.ToString("D4"), x, y, pageCount));
            return nextID;
        }
        
        // Removes event with given ID.
        void removeEvent(int id)
        {
            for (int i = 0; i < events.Count; i++)
            {
                if (events[i].eventNum == id)
                {
                    events.RemoveAt(i);
                    return;
                }
            }
        }
        
        // Checks for teleport commands in all events.
        public void getWarps()
        {
            M.currentFile = filename;
            
            for (int i = 0; i < events.Count; i++)
                events[i].getWarps();
        }
        
        // Saves map file from stored data.
        public bool writeFile(string newFile = "")
        {
            if (newFile == "")
                newFile = filepath;
            
            if (M.fileInUse(newFile))
            {
                Console.WriteLine(filename + " is in use; cannot save.");
                return false;
            }
            
            BinaryWriter mapWriter = new BinaryWriter(new FileStream(newFile, FileMode.Create));
            M.targetWriter = mapWriter;
            
            try
            {
                M.writeString("LcfMapUnit", M.S_CONSTANT);
                
                if (chunks.wasNext(0x01))
                    M.writeLengthMultibyte(tileset);
                if (chunks.wasNext(0x02))
                    M.writeLengthMultibyte(mapWidth);
                if (chunks.wasNext(0x03))
                    M.writeLengthMultibyte(mapHeight);
                
                if (chunks.wasNext(0x0b))
                    M.writeLengthMultibyte(wrapType);
                
                if (chunks.wasNext(0x1f))
                    M.writeLengthBool(useParallax);
                if (chunks.wasNext(0x20))
                    M.writeString(parallaxName, M.S_FILENAME);
                if (chunks.wasNext(0x21))
                    M.writeLengthBool(horzLoop);
                if (chunks.wasNext(0x22))
                    M.writeLengthBool(vertLoop);
                if (chunks.wasNext(0x23))
                    M.writeLengthBool(horzAuto);
                if (chunks.wasNext(0x24))
                    M.writeLengthMultibyte(horzSpeed);
                if (chunks.wasNext(0x25))
                    M.writeLengthBool(vertAuto);
                if (chunks.wasNext(0x26))
                    M.writeLengthMultibyte(vertSpeed);
                
                if (chunks.wasNext(0x28))
                    M.writeLengthBool(useRandomGenerator);
                if (chunks.wasNext(0x29))
                    M.writeLengthMultibyte(generatorMode);
                if (chunks.wasNext(0x2a))
                    M.writeLengthBool(topLevel);
                if (chunks.wasNext(0x30))
                    M.writeLengthMultibyte(generatorGranularity);
                if (chunks.wasNext(0x31))
                    M.writeLengthMultibyte(generatorRoomWidth);
                if (chunks.wasNext(0x32))
                    M.writeLengthMultibyte(generatorRoomHeight);
                if (chunks.wasNext(0x33))
                    M.writeLengthBool(generatorSurroundWithWalls);
                if (chunks.wasNext(0x34))
                    M.writeLengthBool(generatorUseUpperWall);
                if (chunks.wasNext(0x35))
                    M.writeLengthBool(generatorUseFloorB);
                if (chunks.wasNext(0x36))
                    M.writeLengthBool(generatorUseFloorC);
                if (chunks.wasNext(0x37))
                    M.writeLengthBool(generatorUseObstacleB);
                if (chunks.wasNext(0x38))
                    M.writeLengthBool(generatorUseObstacleC);
                
                if (chunks.wasNext(0x3c))
                    M.writeFourByteArray(generatorTileX);
                if (chunks.wasNext(0x3d))
                    M.writeFourByteArray(generatorTileY);
                if (chunks.wasNext(0x3e))
                    M.writeFourByteArray(generatorTileID);
                
                if (chunks.wasNext(0x47))
                    M.writeTwoByteArray2D(layer1Tiles);
                if (chunks.wasNext(0x48))
                    M.writeTwoByteArray2D(layer2Tiles);
                
                if (chunks.wasNext(0x51))
                    M.writeList<Event>(events);
                
                if (chunks.wasNext(0x5b))
                    M.writeLengthMultibyte(saveCount);
                
                M.writeByte(0x00);
                
                mapWriter.Close();
                M.targetWriter.Close();
            }
            catch (Exception e)
            {
                M.debugMessage(e.StackTrace);
                M.debugMessage(e.Message);
                Console.WriteLine("Aborting due to error.");
                
                mapWriter.Close();
                M.targetWriter.Close();
                return false;
            }
            
            return true;
        }
        
        // Returns Event with given ID.
        public Event getEventWithID(int id)
        {
            for (int i = 0; i < events.Count; i++)
            {
                if (events[i].eventNum == id)
                    return events[i];
            }
            
            Console.WriteLine("Warning: Expected event not found! Base game may be missing something.");
            return null;
        }
        
        // Returns Xth command in event and page.
        public Command getCommand(int eventID, int page, int num, int opcode)
        {
            Page editPage = getEventWithID(eventID).getPage(page);
            int commandNum = 0;
            
            for (int i = 0; i < editPage.commands.Count; i++)
            {
                Command command = editPage.commands[i];
                if (command.opcode == opcode)
                {
                    commandNum++;
                    if (commandNum == num)
                        return command;
                }
            }
            
            Console.WriteLine("Warning: Expected command not found! Base game may be missing something.");
            return null;
        }
        
        // Returns Layer 1 tiles.
        public int[][] getTileLayer1()
        {
            return layer1Tiles;
        }
        
        // Replaces Layer 1 tiles.
        public void setTileLayer1(int[][] tiles)
        {
            layer1Tiles = tiles;
        }
        
        // Retrieves version from Game Start map.
        public int baseGameVersion()
        {
            if (id == 2) // Game Start
            {
                Command checkCommand = getCommand(1, 5, 1, Command.C_VARIABLE);
                return checkCommand.args[5];
            }
            
            return -1;
        }
    }
}
